﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using Ozeki.Media;
using Ozeki.VoIP;


namespace VoiceMail.VoiceMail
{
    class VoiceMailPlayCommand : IVoiceMailCommand
    {
        ICall call;
        TextToSpeech textToSpeech;
        PhoneCallAudioSender phoneCallAudioSender;
        string extensionVoiceMailPath;

        MediaConnector mediaConnector;
        WaveStreamPlayback waveStreamPlayback;

        string messageFormat;
        string currentVoiceMailPath;

        private VoiceMailPlayingState playingState;

        public VoiceMailPlayCommand(IPBXCall call, string voiceMailRootPath, string caller)
        {
            extensionVoiceMailPath = Path.Combine(voiceMailRootPath, caller);

            this.call = call;

            mediaConnector = new MediaConnector();
            textToSpeech = new TextToSpeech();
            phoneCallAudioSender = new PhoneCallAudioSender();
            phoneCallAudioSender.AttachToCall(call);

            mediaConnector.Connect(textToSpeech, phoneCallAudioSender);

            playingState = VoiceMailPlayingState.Welcome;
            this.call.CallStateChanged += CallStateChanged;
            textToSpeech.Stopped += TextToSpeechCompleted;
        }

        public event EventHandler Completed;

        public void Execute()
        {
            call.Answer();
        }

        void CallStateChanged(object sender, CallStateChangedArgs e)
        {
            if (e.State == CallState.Answered)
            {
                GotoNextStep();
                return;
            }

            if (e.State.IsCallEnded())
                CleanUp();
        }

        void TextToSpeechCompleted(object sender, EventArgs e)
        {
            List<string> voiceMails;
            switch (playingState)
            {
                case VoiceMailPlayingState.Welcome:
                    voiceMails = GetVoiceMails();
                    if (voiceMails.Count == 0)
                    {
                        playingState = VoiceMailPlayingState.GoodBye;
                        textToSpeech.AddAndStartText("You have no new messages. Good bye, and have a nice day!");
                    }
                    else
                    {
                        textToSpeech.AddAndStartText(string.Format("You have {0} new messages. ", voiceMails.Count));
                        playingState = VoiceMailPlayingState.MessageInfo;
                    }
                        
                    break;

                case VoiceMailPlayingState.MessageInfo:
                    
                    voiceMails = GetVoiceMails();
                    if (voiceMails.Count == 0)
                    {
                        playingState = VoiceMailPlayingState.GoodBye;
                        textToSpeech.AddAndStartText("You have no more new messages. Good bye, and have a nice day!");
                    }
                    else
                    {
                        playingState = VoiceMailPlayingState.Message;
                        PlayVoiceMailInfo();
                    }
         
                    break;

                case VoiceMailPlayingState.Message:
                    voiceMails = GetVoiceMails();
                    if (voiceMails.Count == 0)
                    {
                        playingState = VoiceMailPlayingState.GoodBye;
                        textToSpeech.AddAndStartText("You have no more new messages. Good bye, and have a nice day!");
                    }
                    else
                    {
                        playingState = VoiceMailPlayingState.MessageInfo;
                        PlayVoiceMail();
                    }
                    break;

                case VoiceMailPlayingState.GoodBye:
                    call.HangUp();
                    break;

                default:
                    throw new ArgumentOutOfRangeException();
            }      
        }

        private void PlayVoiceMailInfo()
        {
            currentVoiceMailPath = GetVoiceMails().FirstOrDefault();

            if (currentVoiceMailPath == null)
            {
                playingState = VoiceMailPlayingState.GoodBye;
                textToSpeech.AddAndStartText("You have no more messages. Good bye, and have a nice day!");
                return;
            }

            var fileName = Path.GetFileName(currentVoiceMailPath);
            if (fileName == null)
            {
                Console.WriteLine("Cannot get voicemail filename from path");
                GotoNextStep();
                return;
            }

            var dateFrom =  fileName.Split('_');
            if (fileName.Length < 3)
            {
                Console.WriteLine("Invalid voicemail filename format");
                GotoNextStep();
                return;
            }

            DateTime messageReceivedAt;
            try
            {
                messageReceivedAt = DateTime.ParseExact(dateFrom[0], "yyyy-MM-dd-HH-mm", null);
            }
            catch (Exception)
            {
                Console.WriteLine("Error parsing date in voicemail filename");
                GotoNextStep();
                return;
            }
            var msgDate = messageReceivedAt.ToString("g");
            textToSpeech.AddAndStartText(string.Format("Message received at {0} from {1}. ", msgDate, dateFrom[1]));
        }

        void PlayVoiceMail()
        {
            try
            {
                if (currentVoiceMailPath == null)
                {
                    playingState = VoiceMailPlayingState.GoodBye;
                    textToSpeech.AddAndStartText("You have no more messages. Good bye, and have a nice day!");
                    return;
                }

                waveStreamPlayback = new WaveStreamPlayback(currentVoiceMailPath);
                mediaConnector.Connect(waveStreamPlayback, phoneCallAudioSender);

                waveStreamPlayback.Stopped += VoiceMailPlayingCompleted;
                waveStreamPlayback.Start();
            }
            catch (Exception ex)
            {
                Console.WriteLine("Unexcepted error occurred.", ex);
            }
        }

        void VoiceMailPlayingCompleted(object sender, EventArgs e)
        {
            DeletePlayedVoiceMail();
            GotoNextStep();
        }

        private void GotoNextStep()
        {
            ThreadPool.QueueUserWorkItem(delegate(object state)
                                             { TextToSpeechCompleted(textToSpeech, EventArgs.Empty); });

        }

        void DeletePlayedVoiceMail()
        {
            try
            {
                if (currentVoiceMailPath == null)
                    return;

                if(waveStreamPlayback == null)
                    return;

                mediaConnector.Disconnect(waveStreamPlayback, phoneCallAudioSender);
                waveStreamPlayback.Dispose();

                File.Delete(currentVoiceMailPath);
                currentVoiceMailPath = null;
            }
            catch (Exception ex)
            {
                Console.WriteLine("Unexcepted error occurred.", ex);
            }
        }

        List<string> GetVoiceMails()
        {
            if (!Directory.Exists(extensionVoiceMailPath))
                return new List<string>();

            return new List<string>(Directory.GetFiles(extensionVoiceMailPath, "*.wav"));
        }

        void CleanUp()
        {
            if (waveStreamPlayback != null)
            {
                mediaConnector.Disconnect(waveStreamPlayback, phoneCallAudioSender);
                waveStreamPlayback.Stopped -= VoiceMailPlayingCompleted;
                waveStreamPlayback.Dispose();
            }

            textToSpeech.Dispose();
            mediaConnector.Dispose();
            OnCompleted();
        }

        void OnCompleted()
        {
            var handler = Completed;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private enum VoiceMailPlayingState
        {
            Welcome,
            MessageInfo,
            Message,
            GoodBye
        }
    }
}
