﻿using System;
using System.IO;
using System.Threading;
using Ozeki.Media;
using Ozeki.VoIP;

namespace VoiceMail.VoiceMail
{
    class VoiceMailRecordCommand : IVoiceMailCommand
    {
        ICall call;

        string extensionVoiceMailPath;
        WaveStreamRecorder waveStreamRecorder;
        MediaConnector mediaConnector;
        PhoneCallAudioReceiver phoneCallAudioReceiver;
        PhoneCallAudioSender phoneCallAudioSender;
        TextToSpeech textToSpeech;
        DtmfEventWavePlayer dtmfEventWavePlayer;

        string tempFileName;
        string caller;

        private object sync = new object();
        private string currentDate;

        public VoiceMailRecordCommand(IPBXCall call, string voiceMailRootPath, string target, string caller)
        {
            extensionVoiceMailPath = Path.Combine(voiceMailRootPath, target);
            this.caller = caller;

            mediaConnector = new MediaConnector();

            phoneCallAudioReceiver = new PhoneCallAudioReceiver();
            phoneCallAudioReceiver.AttachToCall(call);

            phoneCallAudioSender = new PhoneCallAudioSender();
            phoneCallAudioSender.AttachToCall(call);

            dtmfEventWavePlayer = new DtmfEventWavePlayer();

            textToSpeech = new TextToSpeech();
            textToSpeech.AddText(string.Format("{0} is not available. Please leave a message after the beep.", target));

            mediaConnector.Connect(textToSpeech, phoneCallAudioSender);
            mediaConnector.Connect(dtmfEventWavePlayer, phoneCallAudioSender);


            this.call = call;
            this.call.CallStateChanged += CallStateChanged;
            textToSpeech.Stopped += TextToSpeechCompleted;
        }

        public event EventHandler Completed;

        public void Execute()
        {
            call.Answer();
        }

        void CallStateChanged(object sender, CallStateChangedArgs e)
        {
            if (e.State.IsInCall())
            {
                textToSpeech.Start();
                return;
            }

            if (e.State.IsCallEnded())
                CleanUp();
        }

        void TextToSpeechCompleted(object sender, EventArgs e)
        {
            SendBeep();
            RecordVoiceMail();
        }

        private void RecordVoiceMail()
        {
            lock (sync)
            {
                try
                {
                    tempFileName = Path.GetTempFileName();

                    waveStreamRecorder = new WaveStreamRecorder(tempFileName);
                    mediaConnector.Connect(phoneCallAudioReceiver, waveStreamRecorder);

                    waveStreamRecorder.Start();
                    currentDate = DateTime.Now.ToString("yyyy-MM-dd-HH-mm");
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Unexcepted error occurred.", ex);
                }
            }
        }

        private void SendBeep()
        {
            dtmfEventWavePlayer.Start(DtmfNamedEvents.Dtmf0);
            Thread.Sleep(500);
            dtmfEventWavePlayer.Stop();
        }

        void CleanUp()
        {
            lock (sync)
            {
                try
                {
                    if (waveStreamRecorder != null)
                    {
                        waveStreamRecorder.Dispose();
                        SaveFile();
                        DeleteTempFile();
                    }

                    textToSpeech.Dispose();
                    dtmfEventWavePlayer.Dispose();
                    mediaConnector.Dispose();

                    OnCompleted();
                }
                catch (Exception ex)
                {
                    Console.WriteLine("Unexpected error occurred: {0}", ex);
                }
            }
        }

        void SaveFile()
        {
            try
            {
                if (tempFileName == null || currentDate == null)
                {
                    Console.WriteLine("Cannot save recorded voicemail, because filename or current date is missing");
                    return;
                }
                    
                if (!Directory.Exists(extensionVoiceMailPath))
                    Directory.CreateDirectory(extensionVoiceMailPath);

                var fileName = string.Format("{0}_{1}_{2}.wav", currentDate, caller, Guid.NewGuid().ToString("N").Substring(0, 8));
                var filePath = Path.Combine(extensionVoiceMailPath, fileName);

                File.Copy(tempFileName, filePath);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Unexpected error occurred: {0}", ex);
            }

        }

        void DeleteTempFile()
        {
            try
            {
                if (tempFileName == null)
                    return;

                File.Delete(tempFileName);
            }
            catch (Exception ex)
            {
                Console.WriteLine("Unexpected error occurred: {0}", ex);
            }

        }

        void OnCompleted()
        {
            var handler = Completed;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }
    }
}
